﻿<#WaaS Info Script CompatScan
#Phase 1 is at start of TS, grabs basic info and writes to registry.
#
#  CompatScan Keys:
#   CompatScanLastRun
#   CompatScanAttempts
#   CompatScanRunTime
#   CompatScanTSRunTime
#   CompatScanDownloadTime
#   CompatScanReturnStatus
#   CompatScanReturnCode
#   CompatScanFailedStepName
#   CompatScanFailedStepReturnCode
#   CompatScanFailedAttempts
#   WaaS_Stage

#NOTE, CompatScanHardBlock & CompatScanVPN are individual Steps in the TS, which is why they are not here.
    
    Changes:
    2021.06.08 - Added Logging into SMSTSlog
#>

$ScriptVer = "21.06.08"
Write-Output "-----------------------------------------"
Write-Output "Starting - Inventory Compat Scan Results Script: $ScriptVer "

#Creates Function to Set how many times the TS Runs
function Set-RegistryValueIncrement {
    [cmdletbinding()]
    param (
        [string] $path,
        [string] $Name
    )

    try { [int]$Value = Get-ItemPropertyValue @PSBoundParameters -ErrorAction SilentlyContinue } catch {}
    Set-ItemProperty @PSBoundParameters -Value ($Value + 1).ToString() 
}

#Setup TS Environment
try
{
    $tsenv = New-Object -COMObject Microsoft.SMS.TSEnvironment
    #SMSTS_Build is set in the Task Sequence
    $tsBuild = $tsenv.Value("SMSTS_Build")
    $registryPath = "HKLM:\$($tsenv.Value("RegistryPath"))\$($tsenv.Value("SMSTS_Build"))"
    Write-Output "Confirmed Running in TS with Registry Path: $registryPath"
}
catch
{
	Write-Verbose " Not running in a task sequence."
}


if ( -not ( test-path $registryPath ) ) { 
    new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null
}

#Gets TS Start Time and Records to Registry
New-ItemProperty -Path $registryPath -Name "CompatScanLastRun" -Value $TSEnv.Value("SMSTS_StartTSTime") -Force | Out-Null
Write-Output " Writing CompatScanLastRun Value to $($TSEnv.Value("SMSTS_StartTSTime"))"

#Increments the ammount of times the Precache CompatScan TS runs
Set-RegistryValueIncrement -Path $registryPath -Name CompatScanAttempts
Write-Output " Writing CompatScanAttempts to $((Get-Item -Path $registrypath).getValue("CompatScanAttempts"))"

#Gets the Time in Minutes it takes to run the CompatScan and Writes to Registry
if ($TSEnv.Value("SMSTS_FinishCSTime") -ne $null -and $TSEnv.Value("SMSTS_FinishCSTime") -ne "")
    {
    $Difference = ([datetime]$TSEnv.Value("SMSTS_FinishCSTime")) - ([datetime]$TSEnv.Value("SMSTS_StartCSTime")) 
    $Difference = [math]::Round($Difference.TotalMinutes)
    if ( -not ( test-path $registryPath ) ) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
    Write-Output " Writing CompatScanRunTime Value to $Difference"
    New-ItemProperty -Path $registryPath -Name "CompatScanRunTime" -Value $Difference -force | Out-Null
    }

#Gets the Time in Minutes it takes to run the CompatScan and Writes to Registry
if ($TSEnv.Value("SMSTS_FinishTSTime") -ne $null -and $TSEnv.Value("SMSTS_FinishTSTime") -ne "")
    {
    $Difference = ([datetime]$TSEnv.Value("SMSTS_FinishTSTime")) - ([datetime]$TSEnv.Value("SMSTS_StartTSTime")) 
    $Difference = [math]::Round($Difference.TotalMinutes)
    if ( -not ( test-path $registryPath ) ) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
    Write-Output " Writing CompatScanTSRunTime Value to $Difference"
    New-ItemProperty -Path $registryPath -Name "CompatScanTSRunTime" -Value $Difference -force | Out-Null
    }

#Gets the Time in Minutes it takes to Download Cache Items and Writes to Registry
if ($TSEnv.Value("SMSTS_FinishTSDownTime") -ne $null -and $TSEnv.Value("SMSTS_FinishTSDownTime") -ne "")
    {
    $DifferenceDown = ([datetime]$TSEnv.Value("SMSTS_FinishTSDownTime")) - ([datetime]$TSEnv.Value("SMSTS_StartTSDownTime")) 
    $DifferenceDown = [math]::Round($DifferenceDown.TotalMinutes)
    if ( -not ( test-path $registryPath ) ) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
    if ((Get-Item -Path $registrypath).getValue("CompatScanDownloadTime") -eq $null) {Write-Output " Writing CompatScanDownloadTime Value to $DifferenceDown"; New-ItemProperty -Path $registryPath -Name "CompatScanDownloadTime" -Value $DifferenceDown -force | Out-Null}
    }

if ($tsenv.Value("_SMSTSOSUpgradeActionReturnCode") -eq $null -or $tsenv.Value("_SMSTSOSUpgradeActionReturnCode") -eq "")
    {
    Write-Output "_SMSTSOSUpgradeActionReturnCode not populated"
    }
else
    {
    #Gets CompatScan Results and Write Code & Friendly Name to Registry
    [int64] $decimalreturncode = $tsenv.Value("_SMSTSOSUpgradeActionReturnCode")
    #[int64] $hexreturncode = 0xC1900210
    $hexreturncode = "{0:X0}" -f [int64]$decimalreturncode

    $WinIPURet = @(
        @{ Err = "C1900210"; Msg = 'No compatibility issues.'}
        @{ Err = "C1900208"; Msg = 'Incompatible apps or drivers.' }
        @{ Err = "C1900204"; Msg = 'Selected migration choice is not available.' }
        @{ Err = "C1900200"; Msg = 'Not eligible for Windows 10.' }
        @{ Err = "C190020E"; Msg = 'Not enough free disk space.' }
        @{ Err = "C1900107"; Msg = 'Unsupported Operating System.' }
        @{ Err = "8024200D"; Msg = 'Update Needs to be Downloaded Again.' }
    )

     $ErrorMsg = $winipuret | ? err -eq $hexreturncode  | % Msg
        
        Write-Output " Writing CompatScanReturnStatus Value to $ErrorMsg"
        New-ItemProperty -path $registryPath -Name "CompatScanReturnStatus" -PropertyType String -Value $ErrorMsg -Force | Out-Null
        #New-ItemProperty -Path $registryPath -Name "CompatScanReturnCodeDec" -Value $tsenv.Value("_SMSTSOSUpgradeActionReturnCode") -force
        Write-Output " Writing CompatScanReturnCode Value to $hexreturncode"
        New-ItemProperty -Path $registryPath -Name "CompatScanReturnCode" -PropertyType String -Value $hexreturncode -force | Out-Null
    }
#Adding key for WaaS Stage
if ( -not ( test-path $registryPath )) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
if ( $hexreturncode -eq "C1900210") 
    {
    if ((Get-Item -Path $registrypath).getValue("CompatScanHardBlock") -ne $null) {Remove-ItemProperty -Path $registrypath -Name CompatScanHardBlock -ErrorAction SilentlyContinue} 
    Write-Output " Writing WaaS_Stage Value to Ready_for_Upgrade"
    New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Ready_for_Upgrade" -force | Out-Null
    }
Else 
    {
    Write-Output " Writing WaaS_Stage Value to Precache_Compat_Scan_Failure"
    New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Precache_Compat_Scan_Failure" -force | Out-Null
    }

<#If Fails / Success 
1) deletes IPURuntime, IPUSetupTime, IPUBuild
2) Increments Failed Attempts
3) Records FailedStepName & LastFailedCode
 
    #>
#Failure
if ($tsenv.Value("AllStepsSucceded") -eq "False")
    {
    Write-Output " Writing CompatScanFailedStepName Value to $($tsenv.Value("FailedStepName"))"
    New-ItemProperty -Path $registryPath -Name "CompatScanFailedStepName" -Value $tsenv.Value("FailedStepName") -force | Out-Null
    Write-Output " Writing CompatScanFailedStepReturnCode Value to $($tsenv.Value("FailedStepReturnCode"))"
    New-ItemProperty -Path $registryPath -Name "CompatScanFailedStepReturnCode" -Value $tsenv.Value("FailedStepReturnCode") -force | Out-Null
    #Add Info if Check Readiness Failed.- Writes FreeSpace to Registry Key CompatScanDiskFreeSpace
    if ($tsenv.Value("FailedStepName") -eq "Check Readiness")
        {
        Get-WmiObject win32_LogicalDisk -Filter "DeviceID='C:'" | % { $FreeSpace = $_.FreeSpace/1GB -as [int] ; $DiskSize = $_.Size/1GB -as [int] }
        Write-Output " Writing CompatScanDiskFreeSpace Value to $FreeSpace"
        New-ItemProperty -Path $registryPath -Name "CompatScanDiskFreeSpace" -Value $FreeSpace -force | Out-Null
        Write-Output " Writing WaaS_Stage Value to Precache_CheckReadiness_Failed"
        New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Precache_CheckReadiness_Failed" -force
        if ($tsenv.Value('CRFailString')){Write-Output " Writing CompatScanReturnStatus Value to $($tsenv.Value('CRFailString'))"; New-ItemProperty -Path $registryPath -Name "CompatScanReturnStatus" -Value $tsenv.Value('CRFailString') -force | Out-Null}
        }
    #Sets CSFailedAttempts Key and Increments after each Failure.
    Set-RegistryValueIncrement -Path $registryPath -Name "CompatScanFailedAttempts"
    Write-Output " Writing CompatScanFailedAttempts to $((Get-Item -Path $registrypath).getValue("CompatScanFailedAttempts"))"
    
    <# - This was for tracking a specific issue and making the reporting more helpful in what the actual error was.
    $SMSTSLog = Get-Content -Path "$($tsenv.Value("_SMSTSLogPath"))\SMSTS.log"
    $Pattern = "Failed to updated security on object"
    if (Select-String -InputObject $SMSTSLog -Pattern $Pattern -Quiet)
        {
        $EDriveFailure = $true
        if ($tsenv.Value("FailedStepName") -eq "Downloading Driver Package")
            {
            New-ItemProperty -path $registryPath -Name "CompatScanReturnStatus" -PropertyType String -Value $Pattern -Force
            }
        }
    #>
    }
#Success
Else
    {
    Write-Output "Additional Cleanup on Successful runs:"
    #Delete CompatScanFailedStepName if exist from previous upgrade (So it doesn't mess with reports)
    if ((Get-Item -Path $registrypath).getValue("CompatScanFailedStepName") -ne $null) {Write-Output "  Removing CompatScanFailedStepName Property"; Remove-ItemProperty -Path $registrypath -Name CompatScanFailedStepName}
    #Delete CompatScanFailedStepReturnCode if exist from previous upgrade (So it doesn't sku results)
    if ((Get-Item -Path $registrypath).getValue("CompatScanFailedStepReturnCode") -ne $null) {Write-Output "  Removing CompatScanFailedStepReturnCode Property"; Remove-ItemProperty -Path $registrypath -Name CompatScanFailedStepReturnCode}
    #Delete IPUFailedStepName if exist from previous upgrade (So it doesn't mess with reports)
    if ((Get-Item -Path $registrypath).getValue("IPUFailedStepName") -ne $null) {Write-Output "  Removing IPUFailedStepName Property"; Remove-ItemProperty -Path $registrypath -Name IPUFailedStepName}
    #Delete IPUFailedStepReturnCode if exist from previous upgrade (So it doesn't sku results)
    if ((Get-Item -Path $registrypath).getValue("IPUFailedStepReturnCode") -ne $null) {Write-Output "  Removing IPUFailedStepReturnCode Property"; Remove-ItemProperty -Path $registrypath -Name IPUFailedStepReturnCode}
    #Delete Free Diskspace Key if exist from previous upgrade (So it doesn't sku results)
    if ((Get-Item -Path $registrypath).getValue("CompatScanDiskFreeSpace") -ne $null) {Write-Output "  Removing CompatScanDiskFreeSpace Property"; Remove-ItemProperty -Path $registrypath -Name CompatScanDiskFreeSpace}
    } 

Write-Output "Completed - Inventory Compat Scan Results Script: $ScriptVer "
Write-Output "-----------------------------------------"
